<?php
/**
 * Trade History Page
 * Bitget Trading Bot
 */

require_once __DIR__ . '/../includes/Auth.php';
require_once __DIR__ . '/../includes/Database.php';

$auth = Auth::getInstance();
$auth->requireAuth();

$db = Database::getInstance();
$userId = $auth->getUserId();

// Pagination
$page = isset($_GET['page']) ? intval($_GET['page']) : 1;
$limit = 20;
$offset = ($page - 1) * $limit;

// Get total count
$totalCount = $db->fetch("SELECT COUNT(*) as count FROM trade_history WHERE user_id = ?", [$userId]);
$totalPages = ceil($totalCount['count'] / $limit);

// Get trade history
$trades = $db->fetchAll(
    "SELECT * FROM trade_history WHERE user_id = ? ORDER BY created_at DESC LIMIT ? OFFSET ?",
    [$userId, $limit, $offset]
);

// Get daily stats
$dailyStats = $db->fetchAll(
    "SELECT * FROM daily_stats WHERE user_id = ? ORDER BY date DESC LIMIT 30",
    [$userId]
);

// Calculate summary
$summary = $db->fetch(
    "SELECT 
        COUNT(*) as total_trades,
        SUM(CASE WHEN status = 'win' THEN 1 ELSE 0 END) as winning_trades,
        SUM(CASE WHEN status = 'loss' THEN 1 ELSE 0 END) as losing_trades,
        SUM(profit_loss) as total_profit,
        AVG(profit_loss_percentage) as avg_profit_pct
    FROM trade_history WHERE user_id = ?",
    [$userId]
);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>History - Bitget Trading Bot</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        body { font-family: 'Inter', sans-serif; }
        .gradient-bg { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); }
        .sidebar-link { transition: all 0.3s ease; }
        .sidebar-link:hover { background: rgba(99, 102, 241, 0.1); color: #6366f1; }
        .sidebar-link.active { background: rgba(99, 102, 241, 0.1); color: #6366f1; border-right: 3px solid #6366f1; }
    </style>
</head>
<body class="bg-gray-50">
    <div class="flex h-screen">
        <!-- Sidebar -->
        <aside class="w-64 bg-white shadow-xl">
            <div class="p-6">
                <div class="flex items-center space-x-3">
                    <div class="w-10 h-10 gradient-bg rounded-lg flex items-center justify-center">
                        <i class="fas fa-robot text-white text-xl"></i>
                    </div>
                    <div>
                        <h1 class="text-xl font-bold text-gray-800">Lompodoe</h1>
                        <p class="text-xs text-gray-500">Trading Bot</p>
                    </div>
                </div>
            </div>
            
            <nav class="mt-6">
                <a href="index.php" class="sidebar-link flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-home w-6"></i>
                    <span>Dashboard</span>
                </a>
                <a href="trading.php" class="sidebar-link flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-chart-line w-6"></i>
                    <span>Trading</span>
                </a>
                <a href="settings.php" class="sidebar-link flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-cog w-6"></i>
                    <span>Settings</span>
                </a>
                <a href="history.php" class="sidebar-link active flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-history w-6"></i>
                    <span>History</span>
                </a>
                <?php if ($auth->isAdmin()): ?>
                <a href="users.php" class="sidebar-link flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-users w-6"></i>
                    <span>Users</span>
                </a>
                <?php endif; ?>
                <a href="logout.php" class="sidebar-link flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-sign-out-alt w-6"></i>
                    <span>Logout</span>
                </a>
            </nav>
        </aside>

        <!-- Main Content -->
        <main class="flex-1 overflow-y-auto">
            <!-- Header -->
            <header class="bg-white shadow-sm">
                <div class="px-8 py-4">
                    <h2 class="text-2xl font-bold text-gray-800">Trade History</h2>
                    <p class="text-gray-500">View your trading performance</p>
                </div>
            </header>

            <!-- Content -->
            <div class="p-8">
                <!-- Summary Cards -->
                <div class="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
                    <div class="bg-white rounded-xl shadow-sm p-6">
                        <p class="text-gray-500 text-sm">Total Trades</p>
                        <p class="text-2xl font-bold text-gray-800"><?php echo $summary['total_trades'] ?? 0; ?></p>
                    </div>
                    <div class="bg-white rounded-xl shadow-sm p-6">
                        <p class="text-gray-500 text-sm">Win Rate</p>
                        <p class="text-2xl font-bold text-blue-600">
                            <?php 
                                $winRate = ($summary['total_trades'] ?? 0) > 0 
                                    ? round(($summary['winning_trades'] / $summary['total_trades']) * 100, 2) 
                                    : 0;
                                echo $winRate;
                            ?>%
                        </p>
                    </div>
                    <div class="bg-white rounded-xl shadow-sm p-6">
                        <p class="text-gray-500 text-sm">Total Profit</p>
                        <p class="text-2xl font-bold <?php echo ($summary['total_profit'] ?? 0) >= 0 ? 'text-green-600' : 'text-red-600'; ?>">
                            <?php echo number_format($summary['total_profit'] ?? 0, 4); ?> USDT
                        </p>
                    </div>
                    <div class="bg-white rounded-xl shadow-sm p-6">
                        <p class="text-gray-500 text-sm">Avg Profit %</p>
                        <p class="text-2xl font-bold text-purple-600">
                            <?php echo number_format($summary['avg_profit_pct'] ?? 0, 2); ?>%
                        </p>
                    </div>
                </div>

                <!-- Profit Chart -->
                <div class="bg-white rounded-xl shadow-sm p-6 mb-8">
                    <h3 class="text-lg font-semibold text-gray-800 mb-4">Daily Profit</h3>
                    <canvas id="profitChart" height="80"></canvas>
                </div>

                <!-- Trade History Table -->
                <div class="bg-white rounded-xl shadow-sm overflow-hidden">
                    <div class="px-6 py-4 border-b">
                        <h3 class="text-lg font-semibold text-gray-800">Trade History</h3>
                    </div>
                    
                    <?php if (empty($trades)): ?>
                        <p class="text-gray-500 text-center py-8">No trade history</p>
                    <?php else: ?>
                        <div class="overflow-x-auto">
                            <table class="w-full">
                                <thead class="bg-gray-50">
                                    <tr class="text-left text-gray-500 text-sm">
                                        <th class="px-6 py-4">Date</th>
                                        <th class="px-6 py-4">Symbol</th>
                                        <th class="px-6 py-4">Side</th>
                                        <th class="px-6 py-4">Entry</th>
                                        <th class="px-6 py-4">Exit</th>
                                        <th class="px-6 py-4">Quantity</th>
                                        <th class="px-6 py-4">P/L</th>
                                        <th class="px-6 py-4">Status</th>
                                    </tr>
                                </thead>
                                <tbody class="divide-y divide-gray-200">
                                    <?php foreach ($trades as $trade): ?>
                                    <tr class="hover:bg-gray-50">
                                        <td class="px-6 py-4 text-sm text-gray-600">
                                            <?php echo date('Y-m-d H:i', strtotime($trade['created_at'])); ?>
                                        </td>
                                        <td class="px-6 py-4 font-medium"><?php echo htmlspecialchars($trade['symbol']); ?></td>
                                        <td class="px-6 py-4">
                                            <span class="px-2 py-1 rounded text-sm <?php echo $trade['side'] === 'buy' ? 'bg-green-100 text-green-700' : 'bg-red-100 text-red-700'; ?>">
                                                <?php echo strtoupper($trade['side']); ?>
                                            </span>
                                        </td>
                                        <td class="px-6 py-4"><?php echo number_format($trade['entry_price'], 4); ?></td>
                                        <td class="px-6 py-4"><?php echo number_format($trade['exit_price'], 4); ?></td>
                                        <td class="px-6 py-4"><?php echo number_format($trade['quantity'], 4); ?></td>
                                        <td class="px-6 py-4 <?php echo $trade['profit_loss'] >= 0 ? 'text-green-600' : 'text-red-600'; ?>">
                                            <?php echo number_format($trade['profit_loss'], 4); ?> USDT
                                        </td>
                                        <td class="px-6 py-4">
                                            <span class="px-2 py-1 rounded text-sm <?php echo $trade['status'] === 'win' ? 'bg-green-100 text-green-700' : 'bg-red-100 text-red-700'; ?>">
                                                <?php echo ucfirst($trade['status']); ?>
                                            </span>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>

                        <!-- Pagination -->
                        <?php if ($totalPages > 1): ?>
                        <div class="px-6 py-4 border-t flex items-center justify-between">
                            <p class="text-sm text-gray-500">
                                Showing <?php echo (($page - 1) * $limit) + 1; ?> - <?php echo min($page * $limit, $totalCount['count']); ?> 
                                of <?php echo $totalCount['count']; ?> trades
                            </p>
                            <div class="flex space-x-2">
                                <?php if ($page > 1): ?>
                                <a href="?page=<?php echo $page - 1; ?>" 
                                    class="px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50">
                                    Previous
                                </a>
                                <?php endif; ?>
                                
                                <?php if ($page < $totalPages): ?>
                                <a href="?page=<?php echo $page + 1; ?>" 
                                    class="px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50">
                                    Next
                                </a>
                                <?php endif; ?>
                            </div>
                        </div>
                        <?php endif; ?>
                    <?php endif; ?>
                </div>
            </div>
        </main>
    </div>

    <script>
        // Daily Profit Chart
        const ctx = document.getElementById('profitChart').getContext('2d');
        
        const dates = <?php echo json_encode(array_column(array_reverse($dailyStats), 'date')); ?>;
        const profits = <?php echo json_encode(array_column(array_reverse($dailyStats), 'net_profit')); ?>;
        
        new Chart(ctx, {
            type: 'bar',
            data: {
                labels: dates,
                datasets: [{
                    label: 'Net Profit (USDT)',
                    data: profits,
                    backgroundColor: profits.map(p => p >= 0 ? 'rgba(16, 185, 129, 0.8)' : 'rgba(239, 68, 68, 0.8)'),
                    borderRadius: 4
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    x: {
                        grid: {
                            display: false
                        }
                    },
                    y: {
                        grid: {
                            color: 'rgba(0, 0, 0, 0.05)'
                        }
                    }
                }
            }
        });
    </script>
</body>
</html>
