<?php
/**
 * Telegram Notification Class
 * Handles all Telegram bot notifications
 */

class Telegram {
    private $botToken;
    private $chatId;
    private $enabled;
    private $lastError;

    public function __construct($botToken = null, $chatId = null, $enabled = false) {
        $this->botToken = $botToken;
        $this->chatId = $chatId;
        $this->enabled = $enabled;
    }

    /**
     * Send message to Telegram
     */
    public function sendMessage($message, $parseMode = 'HTML') {
        if (!$this->enabled || empty($this->botToken) || empty($this->chatId)) {
            return false;
        }

        $url = "https://api.telegram.org/bot{$this->botToken}/sendMessage";
        
        $data = [
            'chat_id' => $this->chatId,
            'text' => $message,
            'parse_mode' => $parseMode,
            'disable_web_page_preview' => true
        ];

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($httpCode !== 200) {
            $this->lastError = "HTTP Error: " . $httpCode;
            return false;
        }

        $result = json_decode($response, true);
        
        if (!isset($result['ok']) || !$result['ok']) {
            $this->lastError = $result['description'] ?? 'Unknown Telegram error';
            return false;
        }

        return true;
    }

    /**
     * Send trade notification
     */
    public function notifyTrade($type, $symbol, $side, $price, $quantity, $total) {
        $emoji = $side === 'buy' ? '🟢' : '🔴';
        $sideText = strtoupper($side);
        
        $message = "{$emoji} <b>TRADE EXECUTED</b> {$emoji}\n\n";
        $message .= "<b>Type:</b> " . strtoupper($type) . "\n";
        $message .= "<b>Symbol:</b> {$symbol}\n";
        $message .= "<b>Side:</b> {$sideText}\n";
        $message .= "<b>Price:</b> {$price}\n";
        $message .= "<b>Quantity:</b> {$quantity}\n";
        $message .= "<b>Total:</b> {$total} USDT\n";
        $message .= "<b>Time:</b> " . date('Y-m-d H:i:s') . "\n";

        return $this->sendMessage($message);
    }

    /**
     * Send profit notification
     */
    public function notifyProfit($symbol, $profit, $profitPercentage, $balance) {
        $emoji = $profit >= 0 ? '💰' : '📉';
        $status = $profit >= 0 ? 'PROFIT' : 'LOSS';
        
        $message = "{$emoji} <b>{$status} REALIZED</b> {$emoji}\n\n";
        $message .= "<b>Symbol:</b> {$symbol}\n";
        $message .= "<b>P/L:</b> " . number_format($profit, 4) . " USDT\n";
        $message .= "<b>P/L %:</b> " . number_format($profitPercentage, 2) . "%\n";
        $message .= "<b>Balance:</b> " . number_format($balance, 4) . " USDT\n";
        $message .= "<b>Time:</b> " . date('Y-m-d H:i:s') . "\n";

        return $this->sendMessage($message);
    }

    /**
     * Send error notification
     */
    public function notifyError($error, $context = '') {
        $message = "⚠️ <b>BOT ERROR</b> ⚠️\n\n";
        $message .= "<b>Error:</b> {$error}\n";
        if (!empty($context)) {
            $message .= "<b>Context:</b> {$context}\n";
        }
        $message .= "<b>Time:</b> " . date('Y-m-d H:i:s') . "\n";

        return $this->sendMessage($message);
    }

    /**
     * Send bot status notification
     */
    public function notifyStatus($status, $details = []) {
        $emoji = $status === 'started' ? '🚀' : ($status === 'stopped' ? '🛑' : 'ℹ️');
        $statusText = strtoupper($status);
        
        $message = "{$emoji} <b>BOT {$statusText}</b> {$emoji}\n\n";
        
        foreach ($details as $key => $value) {
            $message .= "<b>" . ucfirst($key) . ":</b> {$value}\n";
        }
        
        $message .= "<b>Time:</b> " . date('Y-m-d H:i:s') . "\n";

        return $this->sendMessage($message);
    }

    /**
     * Send daily summary
     */
    public function notifyDailySummary($stats) {
        $message = "📊 <b>DAILY TRADING SUMMARY</b> 📊\n\n";
        $message .= "<b>Date:</b> " . $stats['date'] . "\n";
        $message .= "<b>Total Trades:</b> " . $stats['total_trades'] . "\n";
        $message .= "<b>Winning:</b> " . $stats['winning_trades'] . "\n";
        $message .= "<b>Losing:</b> " . $stats['losing_trades'] . "\n";
        $message .= "<b>Net Profit:</b> " . number_format($stats['net_profit'], 4) . " USDT\n";
        $message .= "<b>Balance:</b> " . number_format($stats['balance_end'], 4) . " USDT\n";
        $message .= "<b>Max Drawdown:</b> " . number_format($stats['max_drawdown'], 2) . "%\n";

        return $this->sendMessage($message);
    }

    /**
     * Test connection
     */
    public function testConnection() {
        if (empty($this->botToken)) {
            $this->lastError = 'Bot token is empty';
            return false;
        }

        $url = "https://api.telegram.org/bot{$this->botToken}/getMe";
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($httpCode !== 200) {
            $this->lastError = "HTTP Error: " . $httpCode;
            return false;
        }

        $result = json_decode($response, true);
        return isset($result['ok']) && $result['ok'];
    }

    /**
     * Get last error
     */
    public function getLastError() {
        return $this->lastError;
    }

    /**
     * Set enabled status
     */
    public function setEnabled($enabled) {
        $this->enabled = $enabled;
    }
}
