<?php
/**
 * Bitget API Class
 * Handles all Bitget API interactions
 */

class BitgetAPI {
    private $apiKey;
    private $apiSecret;
    private $passphrase;
    private $isTestnet;
    private $baseUrl;
    private $lastError;
    private $rateLimitRemaining = 10;
    private $lastRequestTime = 0;

    public function __construct($apiKey, $apiSecret, $passphrase, $isTestnet = false) {
        $this->apiKey = $apiKey;
        $this->apiSecret = $apiSecret;
        $this->passphrase = $passphrase;
        $this->isTestnet = $isTestnet;
        $this->baseUrl = BITGET_API_BASE;
    }

    /**
     * Generate signature for Bitget API
     */
    private function generateSignature($timestamp, $method, $requestPath, $body = '') {
        $message = $timestamp . strtoupper($method) . $requestPath . $body;
        $signature = hash_hmac('sha256', $message, $this->apiSecret);
        return $signature;
    }

    /**
     * Make API request with rate limiting
     */
    private function makeRequest($method, $endpoint, $params = [], $isPrivate = false) {
        // Rate limiting
        $currentTime = microtime(true);
        $timeDiff = $currentTime - $this->lastRequestTime;
        
        if ($timeDiff < 0.1) {
            usleep(100000); // 100ms delay
        }

        $url = $this->baseUrl . $endpoint;
        $body = '';
        
        if ($method === 'GET' && !empty($params)) {
            $url .= '?' . http_build_query($params);
        } elseif ($method === 'POST' && !empty($params)) {
            $body = json_encode($params);
        }

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        $headers = ['Content-Type: application/json'];
        
        if ($isPrivate) {
            $timestamp = (string) (time() * 1000);
            $signature = $this->generateSignature($timestamp, $method, $endpoint, $body);
            
            $headers[] = 'ACCESS-KEY: ' . $this->apiKey;
            $headers[] = 'ACCESS-SIGN: ' . $signature;
            $headers[] = 'ACCESS-TIMESTAMP: ' . $timestamp;
            $headers[] = 'ACCESS-PASSPHRASE: ' . $this->passphrase;
        }
        
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        
        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $body);
        }

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        $this->lastRequestTime = microtime(true);

        if ($httpCode !== 200) {
            $this->lastError = "HTTP Error: " . $httpCode;
            return false;
        }

        $data = json_decode($response, true);
        
        if (isset($data['code']) && $data['code'] !== '00000') {
            $this->lastError = $data['msg'] ?? 'Unknown API error';
            return false;
        }

        return $data['data'] ?? $data;
    }

    /**
     * Get account balance
     */
    public function getBalance($coin = 'USDT') {
        $endpoint = '/api/v2/spot/account/assets';
        $params = ['coin' => $coin];
        return $this->makeRequest('GET', $endpoint, $params, true);
    }

    /**
     * Get all USDT trading pairs
     */
    public function getAllSymbols() {
        $endpoint = '/api/v2/spot/public/coins';
        $data = $this->makeRequest('GET', $endpoint, [], false);
        
        if (!$data) return false;
        
        // Filter only USDT pairs
        $symbols = [];
        foreach ($data as $coin) {
            if (isset($coin['coin'])) {
                $symbols[] = $coin['coin'] . 'USDT';
            }
        }
        
        return $symbols;
    }

    /**
     * Get symbol information
     */
    public function getSymbolInfo($symbol) {
        $endpoint = '/api/v2/spot/public/symbols';
        $params = ['symbol' => $symbol];
        return $this->makeRequest('GET', $endpoint, $params, false);
    }

    /**
     * Get current price
     */
    public function getTicker($symbol) {
        $endpoint = '/api/v2/spot/market/tickers';
        $params = ['symbol' => $symbol];
        return $this->makeRequest('GET', $endpoint, $params, false);
    }

    /**
     * Get candlestick data
     */
    public function getKlines($symbol, $granularity = '15m', $limit = 100) {
        $endpoint = '/api/v2/spot/market/candles';
        $params = [
            'symbol' => $symbol,
            'granularity' => $granularity,
            'limit' => $limit
        ];
        return $this->makeRequest('GET', $endpoint, $params, false);
    }

    /**
     * Place order
     */
    public function placeOrder($symbol, $side, $orderType, $size, $price = null) {
        $endpoint = '/api/v2/spot/trade/placeOrder';
        
        $params = [
            'symbol' => $symbol,
            'side' => $side,
            'orderType' => $orderType,
            'size' => (string) $size
        ];
        
        if ($orderType === 'limit' && $price !== null) {
            $params['price'] = (string) $price;
        }
        
        return $this->makeRequest('POST', $endpoint, $params, true);
    }

    /**
     * Cancel order
     */
    public function cancelOrder($symbol, $orderId) {
        $endpoint = '/api/v2/spot/trade/cancelOrder';
        $params = [
            'symbol' => $symbol,
            'orderId' => $orderId
        ];
        return $this->makeRequest('POST', $endpoint, $params, true);
    }

    /**
     * Get order details
     */
    public function getOrder($symbol, $orderId) {
        $endpoint = '/api/v2/spot/trade/orderInfo';
        $params = [
            'symbol' => $symbol,
            'orderId' => $orderId
        ];
        return $this->makeRequest('GET', $endpoint, $params, true);
    }

    /**
     * Get open orders
     */
    public function getOpenOrders($symbol = null) {
        $endpoint = '/api/v2/spot/trade/unfilledOrders';
        $params = [];
        if ($symbol) {
            $params['symbol'] = $symbol;
        }
        return $this->makeRequest('GET', $endpoint, $params, true);
    }

    /**
     * Get order history
     */
    public function getOrderHistory($symbol = null, $limit = 100) {
        $endpoint = '/api/v2/spot/trade/historyOrders';
        $params = ['limit' => $limit];
        if ($symbol) {
            $params['symbol'] = $symbol;
        }
        return $this->makeRequest('GET', $endpoint, $params, true);
    }

    /**
     * Test API connection
     */
    public function testConnection() {
        $result = $this->getBalance('USDT');
        return $result !== false;
    }

    /**
     * Get last error
     */
    public function getLastError() {
        return $this->lastError;
    }

    /**
     * Calculate grid levels
     */
    public function calculateGridLevels($lowerPrice, $upperPrice, $gridCount) {
        $gridSize = ($upperPrice - $lowerPrice) / $gridCount;
        $levels = [];
        
        for ($i = 0; $i <= $gridCount; $i++) {
            $levels[] = $lowerPrice + ($gridSize * $i);
        }
        
        return $levels;
    }

    /**
     * Get current market price
     */
    public function getCurrentPrice($symbol) {
        $ticker = $this->getTicker($symbol);
        if ($ticker && isset($ticker[0]['lastPr'])) {
            return (float) $ticker[0]['lastPr'];
        }
        return false;
    }
}
