<?php
/**
 * Trading Page
 * Bitget Trading Bot - Real-time trading monitor
 */

require_once __DIR__ . '/../includes/Auth.php';
require_once __DIR__ . '/../includes/Database.php';
require_once __DIR__ . '/../includes/BitgetAPI.php';

$auth = Auth::getInstance();
$auth->requireAuth();

$db = Database::getInstance();
$userId = $auth->getUserId();

// Get user settings
$settings = $db->fetch("SELECT * FROM trading_settings WHERE user_id = ?", [$userId]);
$apiSettings = $db->fetch("SELECT * FROM api_settings WHERE user_id = ? AND is_active = 1", [$userId]);

$currentPrice = null;
$balance = null;

// Try to get real data if API is configured
if ($apiSettings) {
    try {
        $api = new BitgetAPI(
            $apiSettings['api_key'],
            $apiSettings['api_secret'],
            $apiSettings['passphrase'],
            $apiSettings['is_testnet']
        );
        
        $currentPrice = $api->getCurrentPrice($settings['symbol']);
        $balanceData = $api->getBalance('USDT');
        if ($balanceData && isset($balanceData[0]['available'])) {
            $balance = $balanceData[0]['available'];
        }
    } catch (Exception $e) {
        // API not configured or error
    }
}

// Get grid orders
$gridOrders = $db->fetchAll("SELECT * FROM grid_orders WHERE user_id = ? ORDER BY grid_level ASC", [$userId]);

// Get DCA orders
$dcaOrders = $db->fetchAll("SELECT * FROM dca_orders WHERE user_id = ? ORDER BY dca_level ASC", [$userId]);

// Get active trades
$activeTrades = $db->fetchAll("SELECT * FROM active_trades WHERE user_id = ? AND status = 'open' ORDER BY created_at DESC", [$userId]);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Trading - Bitget Trading Bot</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        body { font-family: 'Inter', sans-serif; }
        .gradient-bg { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); }
        .sidebar-link { transition: all 0.3s ease; }
        .sidebar-link:hover { background: rgba(99, 102, 241, 0.1); color: #6366f1; }
        .sidebar-link.active { background: rgba(99, 102, 241, 0.1); color: #6366f1; border-right: 3px solid #6366f1; }
        .price-up { color: #10b981; }
        .price-down { color: #ef4444; }
        @keyframes pulse-dot {
            0%, 100% { transform: scale(1); opacity: 1; }
            50% { transform: scale(1.2); opacity: 0.7; }
        }
        .live-indicator {
            animation: pulse-dot 2s infinite;
        }
    </style>
</head>
<body class="bg-gray-50">
    <div class="flex h-screen">
        <!-- Sidebar -->
        <aside class="w-64 bg-white shadow-xl">
            <div class="p-6">
                <div class="flex items-center space-x-3">
                    <div class="w-10 h-10 gradient-bg rounded-lg flex items-center justify-center">
                        <i class="fas fa-robot text-white text-xl"></i>
                    </div>
                    <div>
                        <h1 class="text-xl font-bold text-gray-800">Lompodoe</h1>
                        <p class="text-xs text-gray-500">Trading Bot</p>
                    </div>
                </div>
            </div>
            
            <nav class="mt-6">
                <a href="index.php" class="sidebar-link flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-home w-6"></i>
                    <span>Dashboard</span>
                </a>
                <a href="trading.php" class="sidebar-link active flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-chart-line w-6"></i>
                    <span>Trading</span>
                </a>
                <a href="settings.php" class="sidebar-link flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-cog w-6"></i>
                    <span>Settings</span>
                </a>
                <a href="history.php" class="sidebar-link flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-history w-6"></i>
                    <span>History</span>
                </a>
                <?php if ($auth->isAdmin()): ?>
                <a href="users.php" class="sidebar-link flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-users w-6"></i>
                    <span>Users</span>
                </a>
                <?php endif; ?>
                <a href="logout.php" class="sidebar-link flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-sign-out-alt w-6"></i>
                    <span>Logout</span>
                </a>
            </nav>
        </aside>

        <!-- Main Content -->
        <main class="flex-1 overflow-y-auto">
            <!-- Header -->
            <header class="bg-white shadow-sm">
                <div class="flex items-center justify-between px-8 py-4">
                    <div>
                        <h2 class="text-2xl font-bold text-gray-800">Trading Monitor</h2>
                        <p class="text-gray-500">Real-time trading activity</p>
                    </div>
                    <div class="flex items-center space-x-4">
                        <div class="flex items-center space-x-2 bg-green-50 px-4 py-2 rounded-lg">
                            <div class="w-2 h-2 bg-green-500 rounded-full live-indicator"></div>
                            <span class="text-green-700 font-medium">Live</span>
                        </div>
                    </div>
                </div>
            </header>

            <!-- Content -->
            <div class="p-8">
                <!-- Price & Balance Cards -->
                <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
                    <div class="bg-white rounded-xl shadow-sm p-6">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-gray-500 text-sm">Current Price</p>
                                <p class="text-2xl font-bold text-gray-800">
                                    <?php echo $currentPrice ? number_format($currentPrice, 2) : '--'; ?> USDT
                                </p>
                                <p class="text-sm text-gray-500"><?php echo htmlspecialchars($settings['symbol']); ?></p>
                            </div>
                            <div class="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center">
                                <i class="fas fa-dollar-sign text-blue-600 text-xl"></i>
                            </div>
                        </div>
                    </div>

                    <div class="bg-white rounded-xl shadow-sm p-6">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-gray-500 text-sm">Balance</p>
                                <p class="text-2xl font-bold text-gray-800">
                                    <?php echo $balance ? number_format($balance, 4) : ($settings['is_test_mode'] ? number_format($settings['test_balance'], 4) : '--'); ?> USDT
                                </p>
                                <p class="text-sm text-gray-500"><?php echo $settings['is_test_mode'] ? 'Test Mode' : 'Live'; ?></p>
                            </div>
                            <div class="w-12 h-12 bg-green-100 rounded-lg flex items-center justify-center">
                                <i class="fas fa-wallet text-green-600 text-xl"></i>
                            </div>
                        </div>
                    </div>

                    <div class="bg-white rounded-xl shadow-sm p-6">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-gray-500 text-sm">Active Positions</p>
                                <p class="text-2xl font-bold text-gray-800"><?php echo count($activeTrades); ?></p>
                                <p class="text-sm text-gray-500">Open trades</p>
                            </div>
                            <div class="w-12 h-12 bg-purple-100 rounded-lg flex items-center justify-center">
                                <i class="fas fa-chart-bar text-purple-600 text-xl"></i>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Price Chart -->
                <div class="bg-white rounded-xl shadow-sm p-6 mb-8">
                    <div class="flex items-center justify-between mb-4">
                        <h3 class="text-lg font-semibold text-gray-800">Price Chart</h3>
                        <div class="flex items-center space-x-2">
                            <span class="text-sm text-gray-500">Timeframe:</span>
                            <span class="px-3 py-1 bg-indigo-100 text-indigo-700 rounded-lg text-sm font-medium">
                                <?php echo strtoupper($settings['timeframe']); ?>
                            </span>
                        </div>
                    </div>
                    <canvas id="priceChart" height="80"></canvas>
                </div>

                <!-- Grid Orders & DCA Orders -->
                <div class="grid grid-cols-1 lg:grid-cols-2 gap-8 mb-8">
                    <!-- Grid Orders -->
                    <div class="bg-white rounded-xl shadow-sm p-6">
                        <div class="flex items-center justify-between mb-4">
                            <h3 class="text-lg font-semibold text-gray-800">Grid Orders</h3>
                            <span class="px-3 py-1 <?php echo $settings['mode_grid'] ? 'bg-green-100 text-green-700' : 'bg-gray-100 text-gray-600'; ?> rounded-lg text-sm">
                                <?php echo $settings['mode_grid'] ? 'Active' : 'Inactive'; ?>
                            </span>
                        </div>
                        
                        <?php if (empty($gridOrders)): ?>
                            <p class="text-gray-500 text-center py-8">No grid orders</p>
                        <?php else: ?>
                            <div class="overflow-x-auto">
                                <table class="w-full">
                                    <thead>
                                        <tr class="text-left text-gray-500 text-sm border-b">
                                            <th class="pb-3">Level</th>
                                            <th class="pb-3">Price</th>
                                            <th class="pb-3">Side</th>
                                            <th class="pb-3">Status</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($gridOrders as $order): ?>
                                        <tr class="border-b last:border-0">
                                            <td class="py-3"><?php echo $order['grid_level']; ?></td>
                                            <td class="py-3 font-medium"><?php echo number_format($order['price'], 4); ?></td>
                                            <td class="py-3">
                                                <span class="px-2 py-1 rounded text-sm <?php echo $order['side'] === 'buy' ? 'bg-green-100 text-green-700' : 'bg-red-100 text-red-700'; ?>">
                                                    <?php echo strtoupper($order['side']); ?>
                                                </span>
                                            </td>
                                            <td class="py-3">
                                                <span class="px-2 py-1 rounded text-sm <?php 
                                                    echo $order['status'] === 'filled' ? 'bg-green-100 text-green-700' : 
                                                         ($order['status'] === 'pending' ? 'bg-yellow-100 text-yellow-700' : 'bg-gray-100 text-gray-600'); 
                                                ?>">
                                                    <?php echo ucfirst($order['status']); ?>
                                                </span>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>

                    <!-- DCA Orders -->
                    <div class="bg-white rounded-xl shadow-sm p-6">
                        <div class="flex items-center justify-between mb-4">
                            <h3 class="text-lg font-semibold text-gray-800">DCA Levels</h3>
                            <span class="px-3 py-1 <?php echo $settings['mode_dca'] ? 'bg-green-100 text-green-700' : 'bg-gray-100 text-gray-600'; ?> rounded-lg text-sm">
                                <?php echo $settings['mode_dca'] ? 'Active' : 'Inactive'; ?>
                            </span>
                        </div>
                        
                        <?php if (empty($dcaOrders)): ?>
                            <p class="text-gray-500 text-center py-8">No DCA orders</p>
                        <?php else: ?>
                            <div class="overflow-x-auto">
                                <table class="w-full">
                                    <thead>
                                        <tr class="text-left text-gray-500 text-sm border-b">
                                            <th class="pb-3">Level</th>
                                            <th class="pb-3">Trigger Price</th>
                                            <th class="pb-3">Quantity</th>
                                            <th class="pb-3">Status</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($dcaOrders as $order): ?>
                                        <tr class="border-b last:border-0">
                                            <td class="py-3"><?php echo $order['dca_level']; ?></td>
                                            <td class="py-3 font-medium"><?php echo number_format($order['trigger_price'], 4); ?></td>
                                            <td class="py-3"><?php echo number_format($order['quantity'], 4); ?></td>
                                            <td class="py-3">
                                                <span class="px-2 py-1 rounded text-sm <?php 
                                                    echo $order['status'] === 'filled' ? 'bg-green-100 text-green-700' : 
                                                         ($order['status'] === 'pending' ? 'bg-yellow-100 text-yellow-700' : 'bg-gray-100 text-gray-600'); 
                                                ?>">
                                                    <?php echo ucfirst($order['status']); ?>
                                                </span>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Active Trades -->
                <div class="bg-white rounded-xl shadow-sm p-6">
                    <h3 class="text-lg font-semibold text-gray-800 mb-4">Active Trades</h3>
                    
                    <?php if (empty($activeTrades)): ?>
                        <p class="text-gray-500 text-center py-8">No active trades</p>
                    <?php else: ?>
                        <div class="overflow-x-auto">
                            <table class="w-full">
                                <thead>
                                    <tr class="text-left text-gray-500 text-sm border-b">
                                        <th class="pb-3">Symbol</th>
                                        <th class="pb-3">Side</th>
                                        <th class="pb-3">Entry Price</th>
                                        <th class="pb-3">Quantity</th>
                                        <th class="pb-3">Total</th>
                                        <th class="pb-3">Type</th>
                                        <th class="pb-3">Time</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($activeTrades as $trade): ?>
                                    <tr class="border-b last:border-0">
                                        <td class="py-3 font-medium"><?php echo htmlspecialchars($trade['symbol']); ?></td>
                                        <td class="py-3">
                                            <span class="px-2 py-1 rounded text-sm <?php echo $trade['side'] === 'buy' ? 'bg-green-100 text-green-700' : 'bg-red-100 text-red-700'; ?>">
                                                <?php echo strtoupper($trade['side']); ?>
                                            </span>
                                        </td>
                                        <td class="py-3"><?php echo number_format($trade['entry_price'], 4); ?></td>
                                        <td class="py-3"><?php echo number_format($trade['quantity'], 4); ?></td>
                                        <td class="py-3"><?php echo number_format($trade['total_amount'], 4); ?> USDT</td>
                                        <td class="py-3">
                                            <span class="px-2 py-1 bg-blue-100 text-blue-700 rounded text-sm uppercase">
                                                <?php echo $trade['trade_type']; ?>
                                            </span>
                                        </td>
                                        <td class="py-3 text-gray-500 text-sm">
                                            <?php echo date('Y-m-d H:i', strtotime($trade['created_at'])); ?>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </main>
    </div>

    <script>
        // Simulated price chart
        const ctx = document.getElementById('priceChart').getContext('2d');
        
        // Generate some sample data
        const labels = [];
        const data = [];
        let basePrice = <?php echo $currentPrice ? $currentPrice : 50000; ?>;
        
        for (let i = 0; i < 50; i++) {
            labels.push(i);
            basePrice = basePrice * (1 + (Math.random() - 0.5) * 0.002);
            data.push(basePrice);
        }
        
        new Chart(ctx, {
            type: 'line',
            data: {
                labels: labels,
                datasets: [{
                    label: 'Price',
                    data: data,
                    borderColor: '#6366f1',
                    backgroundColor: 'rgba(99, 102, 241, 0.1)',
                    borderWidth: 2,
                    tension: 0.4,
                    fill: true,
                    pointRadius: 0
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    x: {
                        display: false
                    },
                    y: {
                        grid: {
                            color: 'rgba(0, 0, 0, 0.05)'
                        }
                    }
                },
                interaction: {
                    intersect: false,
                    mode: 'index'
                }
            }
        });

        // Auto-refresh page every 30 seconds
        setTimeout(() => {
            window.location.reload();
        }, 30000);
    </script>
</body>
</html>
