<?php
/**
 * Authentication Class
 * Handles user login, logout, and session management
 */

require_once __DIR__ . '/Database.php';

class Auth {
    private static $instance = null;
    private $db;
    private $user = null;

    private function __construct() {
        $this->db = Database::getInstance();
        $this->startSession();
        $this->checkSession();
    }

    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Start secure session
     */
    private function startSession() {
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }
    }

    /**
     * Check and validate session
     */
    private function checkSession() {
        if (isset($_SESSION['user_id']) && isset($_SESSION['last_activity'])) {
            // Check session timeout
            if (time() - $_SESSION['last_activity'] > SESSION_TIMEOUT) {
                $this->logout();
                return;
            }

            // Update last activity
            $_SESSION['last_activity'] = time();

            // Load user data
            $this->loadUser($_SESSION['user_id']);
        }
    }

    /**
     * Load user data
     */
    private function loadUser($userId) {
        $sql = "SELECT id, username, email, role, is_active, last_login FROM users WHERE id = ? LIMIT 1";
        $this->user = $this->db->fetch($sql, [$userId]);
    }

    /**
     * Login user
     */
    public function login($username, $password) {
        $sql = "SELECT id, username, password, role, is_active FROM users WHERE username = ? OR email = ? LIMIT 1";
        $user = $this->db->fetch($sql, [$username, $username]);

        if (!$user) {
            return ['success' => false, 'message' => 'Invalid credentials'];
        }

        if (!$user['is_active']) {
            return ['success' => false, 'message' => 'Account is disabled'];
        }

        if (!password_verify($password, $user['password'])) {
            return ['success' => false, 'message' => 'Invalid credentials'];
        }

        // Set session
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['username'] = $user['username'];
        $_SESSION['role'] = $user['role'];
        $_SESSION['last_activity'] = time();

        // Update last login
        $this->db->update('users',
            ['last_login' => date('Y-m-d H:i:s')],
            'id = :id',
            ['id' => $user['id']]
        );

        $this->loadUser($user['id']);

        return ['success' => true, 'message' => 'Login successful'];
    }

    /**
     * Logout user
     */
    public function logout() {
        session_unset();
        session_destroy();
        $this->user = null;
    }

    /**
     * Check if user is logged in
     */
    public function isLoggedIn() {
        return $this->user !== null;
    }

    /**
     * Check if user is admin
     */
    public function isAdmin() {
        return $this->isLoggedIn() && $this->user['role'] === 'admin';
    }

    /**
     * Get current user
     */
    public function getUser() {
        return $this->user;
    }

    /**
     * Get user ID
     */
    public function getUserId() {
        return $this->user ? $this->user['id'] : null;
    }

    /**
     * Require authentication
     */
    public function requireAuth() {
        if (!$this->isLoggedIn()) {
            header('Location: /login.php');
            exit;
        }
    }

    /**
     * Require admin
     */
    public function requireAdmin() {
        $this->requireAuth();
        if (!$this->isAdmin()) {
            header('Location: /dashboard.php');
            exit;
        }
    }

    /**
     * Register new user (admin only)
     */
    public function register($username, $email, $password, $role = 'user') {
        // Check if username exists
        $sql = "SELECT id FROM users WHERE username = ? OR email = ? LIMIT 1";
        $existing = $this->db->fetch($sql, [$username, $email]);

        if ($existing) {
            return ['success' => false, 'message' => 'Username or email already exists'];
        }

        // Hash password
        $hashedPassword = password_hash($password, PASSWORD_BCRYPT);

        // Insert user
        $userId = $this->db->insert('users', [
            'username' => $username,
            'email' => $email,
            'password' => $hashedPassword,
            'role' => $role
        ]);

        if ($userId) {
            // Create default settings
            $this->db->insert('trading_settings', ['user_id' => $userId]);
            $this->db->insert('telegram_settings', ['user_id' => $userId]);

            return ['success' => true, 'message' => 'User created successfully', 'user_id' => $userId];
        }

        return ['success' => false, 'message' => 'Failed to create user'];
    }

    /**
     * Update user
     */
    public function updateUser($userId, $data) {
        $allowedFields = ['username', 'email', 'is_active'];
        $updateData = [];

        foreach ($data as $key => $value) {
            if (in_array($key, $allowedFields)) {
                $updateData[$key] = $value;
            }
        }

        if (empty($updateData)) {
            return ['success' => false, 'message' => 'No valid fields to update'];
        }

        $result = $this->db->update('users', $updateData, 'id = :id', ['id' => $userId]);

        return $result ? 
            ['success' => true, 'message' => 'User updated successfully'] : 
            ['success' => false, 'message' => 'Failed to update user'];
    }

    /**
     * Change password
     */
    public function changePassword($userId, $oldPassword, $newPassword) {
        // Verify old password
        $sql = "SELECT password FROM users WHERE id = ? LIMIT 1";
        $user = $this->db->fetch($sql, [$userId]);

        if (!$user || !password_verify($oldPassword, $user['password'])) {
            return ['success' => false, 'message' => 'Current password is incorrect'];
        }

        // Update password
        $hashedPassword = password_hash($newPassword, PASSWORD_BCRYPT);
        $result = $this->db->update('users',
            ['password' => $hashedPassword],
            'id = :id',
            ['id' => $userId]
        );

        return $result ?
            ['success' => true, 'message' => 'Password changed successfully'] :
            ['success' => false, 'message' => 'Failed to change password'];
    }

    /**
     * Get all users (admin only)
     */
    public function getAllUsers() {
        $sql = "SELECT id, username, email, role, is_active, created_at, last_login FROM users ORDER BY created_at DESC";
        return $this->db->fetchAll($sql);
    }

    /**
     * Delete user (admin only)
     */
    public function deleteUser($userId) {
        // Prevent deleting self
        if ($userId == $this->getUserId()) {
            return ['success' => false, 'message' => 'Cannot delete your own account'];
        }

        $result = $this->db->delete('users', 'id = ?', [$userId]);

        return $result ?
            ['success' => true, 'message' => 'User deleted successfully'] :
            ['success' => false, 'message' => 'Failed to delete user'];
    }
}
